/*
 * Decompiled with CFR 0.152.
 */
package net.minecraft.data;

import com.google.common.collect.ImmutableMap;
import com.google.common.hash.HashCode;
import com.google.common.hash.Hashing;
import com.mojang.logging.LogUtils;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicInteger;
import javax.annotation.Nullable;
import net.minecraft.WorldVersion;
import net.minecraft.data.CachedOutput;
import org.apache.commons.lang3.mutable.MutableInt;
import org.slf4j.Logger;

public class HashCache {
    static final Logger LOGGER = LogUtils.getLogger();
    private static final String HEADER_MARKER = "// ";
    private final Path rootDir;
    private final Path cacheDir;
    private final String versionId;
    private final Map<String, ProviderCache> caches;
    private final Set<String> cachesToWrite = new HashSet<String>();
    final Set<Path> cachePaths = new HashSet<Path>();
    private final int initialCount;
    private int writes;

    private Path getProviderCachePath(String provider) {
        return this.cacheDir.resolve(Hashing.sha1().hashString((CharSequence)provider, StandardCharsets.UTF_8).toString());
    }

    public HashCache(Path rootDir, Collection<String> providers, WorldVersion version) throws IOException {
        this.versionId = version.id();
        this.rootDir = rootDir;
        this.cacheDir = rootDir.resolve(".cache");
        Files.createDirectories(this.cacheDir, new FileAttribute[0]);
        HashMap<String, ProviderCache> map = new HashMap<String, ProviderCache>();
        int i = 0;
        for (String string : providers) {
            Path providerCachePath = this.getProviderCachePath(string);
            this.cachePaths.add(providerCachePath);
            ProviderCache cache = HashCache.readCache(rootDir, providerCachePath);
            map.put(string, cache);
            i += cache.count();
        }
        this.caches = map;
        this.initialCount = i;
    }

    private static ProviderCache readCache(Path rootDir, Path cachePath) {
        if (Files.isReadable(cachePath)) {
            try {
                return ProviderCache.load(rootDir, cachePath);
            }
            catch (Exception var3) {
                LOGGER.warn("Failed to parse cache {}, discarding", (Object)cachePath, (Object)var3);
            }
        }
        return new ProviderCache("unknown", (ImmutableMap<Path, HashCode>)ImmutableMap.of());
    }

    public boolean shouldRunInThisVersion(String provider) {
        ProviderCache providerCache = this.caches.get(provider);
        return providerCache == null || !providerCache.version.equals(this.versionId);
    }

    public CompletableFuture<UpdateResult> generateUpdate(String provider, UpdateFunction updateFunction) {
        ProviderCache providerCache = this.caches.get(provider);
        if (providerCache == null) {
            throw new IllegalStateException("Provider not registered: " + provider);
        }
        CacheUpdater cacheUpdater = new CacheUpdater(provider, this.versionId, providerCache);
        return updateFunction.update(cacheUpdater).thenApply(object -> cacheUpdater.close());
    }

    public void applyUpdate(UpdateResult updateResult) {
        this.caches.put(updateResult.providerId(), updateResult.cache());
        this.cachesToWrite.add(updateResult.providerId());
        this.writes += updateResult.writes();
    }

    public void purgeStaleAndWrite() throws IOException {
        final HashSet<Path> set = new HashSet<Path>();
        this.caches.forEach((cacheName, providerCache) -> {
            if (this.cachesToWrite.contains(cacheName)) {
                Path providerCachePath = this.getProviderCachePath((String)cacheName);
                providerCache.save(this.rootDir, providerCachePath, DateTimeFormatter.ISO_LOCAL_DATE_TIME.format(LocalDateTime.now()) + "\t" + cacheName);
            }
            set.addAll((Collection<Path>)providerCache.data().keySet());
        });
        set.add(this.rootDir.resolve("version.json"));
        final MutableInt mutableInt = new MutableInt();
        final MutableInt mutableInt1 = new MutableInt();
        Files.walkFileTree(this.rootDir, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path path, BasicFileAttributes basicFileAttributes) {
                if (HashCache.this.cachePaths.contains(path)) {
                    return FileVisitResult.CONTINUE;
                }
                mutableInt.increment();
                if (set.contains(path)) {
                    return FileVisitResult.CONTINUE;
                }
                try {
                    Files.delete(path);
                }
                catch (IOException var4) {
                    LOGGER.warn("Failed to delete file {}", (Object)path, (Object)var4);
                }
                mutableInt1.increment();
                return FileVisitResult.CONTINUE;
            }
        });
        LOGGER.info("Caching: total files: {}, old count: {}, new count: {}, removed stale: {}, written: {}", new Object[]{mutableInt, this.initialCount, set.size(), mutableInt1, this.writes});
    }

    record ProviderCache(String version, ImmutableMap<Path, HashCode> data) {
        @Nullable
        public HashCode get(Path path) {
            return (HashCode)this.data.get((Object)path);
        }

        public int count() {
            return this.data.size();
        }

        public static ProviderCache load(Path rootDir, Path cachePath) throws IOException {
            ProviderCache var7;
            try (BufferedReader bufferedReader = Files.newBufferedReader(cachePath, StandardCharsets.UTF_8);){
                String line = bufferedReader.readLine();
                if (!line.startsWith(HashCache.HEADER_MARKER)) {
                    throw new IllegalStateException("Missing cache file header");
                }
                String[] parts = line.substring(HashCache.HEADER_MARKER.length()).split("\t", 2);
                String string = parts[0];
                ImmutableMap.Builder builder = ImmutableMap.builder();
                bufferedReader.lines().forEach(line1 -> {
                    int index = line1.indexOf(32);
                    builder.put((Object)rootDir.resolve(line1.substring(index + 1)), (Object)HashCode.fromString((String)line1.substring(0, index)));
                });
                var7 = new ProviderCache(string, (ImmutableMap<Path, HashCode>)builder.build());
            }
            return var7;
        }

        public void save(Path rootDir, Path cachePath, String date) {
            try (BufferedWriter bufferedWriter = Files.newBufferedWriter(cachePath, StandardCharsets.UTF_8, new OpenOption[0]);){
                bufferedWriter.write(HashCache.HEADER_MARKER);
                bufferedWriter.write(this.version);
                bufferedWriter.write(9);
                bufferedWriter.write(date);
                bufferedWriter.newLine();
                for (Map.Entry entry : this.data.entrySet()) {
                    bufferedWriter.write(((HashCode)entry.getValue()).toString());
                    bufferedWriter.write(32);
                    bufferedWriter.write(rootDir.relativize((Path)entry.getKey()).toString());
                    bufferedWriter.newLine();
                }
            }
            catch (IOException var9) {
                LOGGER.warn("Unable write cachefile {}: {}", (Object)cachePath, (Object)var9);
            }
        }
    }

    static class CacheUpdater
    implements CachedOutput {
        private final String provider;
        private final ProviderCache oldCache;
        private final ProviderCacheBuilder newCache;
        private final AtomicInteger writes = new AtomicInteger();
        private volatile boolean closed;

        CacheUpdater(String provider, String version, ProviderCache oldCache) {
            this.provider = provider;
            this.oldCache = oldCache;
            this.newCache = new ProviderCacheBuilder(version);
        }

        private boolean shouldWrite(Path key, HashCode value) {
            return !Objects.equals(this.oldCache.get(key), value) || !Files.exists(key, new LinkOption[0]);
        }

        @Override
        public void writeIfNeeded(Path filePath, byte[] data, HashCode hashCode) throws IOException {
            if (this.closed) {
                throw new IllegalStateException("Cannot write to cache as it has already been closed");
            }
            if (this.shouldWrite(filePath, hashCode)) {
                this.writes.incrementAndGet();
                Files.createDirectories(filePath.getParent(), new FileAttribute[0]);
                Files.write(filePath, data, new OpenOption[0]);
            }
            this.newCache.put(filePath, hashCode);
        }

        public UpdateResult close() {
            this.closed = true;
            return new UpdateResult(this.provider, this.newCache.build(), this.writes.get());
        }
    }

    @FunctionalInterface
    public static interface UpdateFunction {
        public CompletableFuture<?> update(CachedOutput var1);
    }

    public record UpdateResult(String providerId, ProviderCache cache, int writes) {
    }

    record ProviderCacheBuilder(String version, ConcurrentMap<Path, HashCode> data) {
        ProviderCacheBuilder(String version) {
            this(version, new ConcurrentHashMap<Path, HashCode>());
        }

        public void put(Path key, HashCode value) {
            this.data.put(key, value);
        }

        public ProviderCache build() {
            return new ProviderCache(this.version, (ImmutableMap<Path, HashCode>)ImmutableMap.copyOf(this.data));
        }
    }
}

