/*
 * Decompiled with CFR 0.152.
 */
package net.minecraft.world.waypoints;

import com.mojang.datafixers.util.Either;
import com.mojang.logging.LogUtils;
import io.netty.buffer.ByteBuf;
import java.util.UUID;
import net.minecraft.core.UUIDUtil;
import net.minecraft.core.Vec3i;
import net.minecraft.network.FriendlyByteBuf;
import net.minecraft.network.VarInt;
import net.minecraft.network.codec.StreamCodec;
import net.minecraft.util.Mth;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.level.ChunkPos;
import net.minecraft.world.level.Level;
import net.minecraft.world.phys.Vec3;
import net.minecraft.world.waypoints.PartialTickSupplier;
import net.minecraft.world.waypoints.Waypoint;
import org.apache.commons.lang3.function.TriFunction;
import org.slf4j.Logger;

public abstract class TrackedWaypoint
implements Waypoint {
    static final Logger LOGGER = LogUtils.getLogger();
    public static final StreamCodec<ByteBuf, TrackedWaypoint> STREAM_CODEC = StreamCodec.ofMember(TrackedWaypoint::write, TrackedWaypoint::read);
    protected final Either<UUID, String> identifier;
    private final Waypoint.Icon icon;
    private final Type type;

    TrackedWaypoint(Either<UUID, String> identifier, Waypoint.Icon icon, Type type) {
        this.identifier = identifier;
        this.icon = icon;
        this.type = type;
    }

    public Either<UUID, String> id() {
        return this.identifier;
    }

    public abstract void update(TrackedWaypoint var1);

    public void write(ByteBuf buffer) {
        FriendlyByteBuf friendlyByteBuf = new FriendlyByteBuf(buffer);
        friendlyByteBuf.writeEither(this.identifier, UUIDUtil.STREAM_CODEC, FriendlyByteBuf::writeUtf);
        Waypoint.Icon.STREAM_CODEC.encode(friendlyByteBuf, this.icon);
        friendlyByteBuf.writeEnum(this.type);
        this.writeContents(buffer);
    }

    public abstract void writeContents(ByteBuf var1);

    private static TrackedWaypoint read(ByteBuf buffer) {
        FriendlyByteBuf friendlyByteBuf = new FriendlyByteBuf(buffer);
        Either<UUID, String> either = friendlyByteBuf.readEither(UUIDUtil.STREAM_CODEC, FriendlyByteBuf::readUtf);
        Waypoint.Icon icon = (Waypoint.Icon)Waypoint.Icon.STREAM_CODEC.decode(friendlyByteBuf);
        Type type = friendlyByteBuf.readEnum(Type.class);
        return (TrackedWaypoint)type.constructor.apply(either, (Object)icon, (Object)friendlyByteBuf);
    }

    public static TrackedWaypoint setPosition(UUID uuid, Waypoint.Icon icon, Vec3i position) {
        return new Vec3iWaypoint(uuid, icon, position);
    }

    public static TrackedWaypoint setChunk(UUID uuid, Waypoint.Icon icon, ChunkPos chunkPos) {
        return new ChunkWaypoint(uuid, icon, chunkPos);
    }

    public static TrackedWaypoint setAzimuth(UUID uuid, Waypoint.Icon icon, float angle) {
        return new AzimuthWaypoint(uuid, icon, angle);
    }

    public static TrackedWaypoint empty(UUID uuid) {
        return new EmptyWaypoint(uuid);
    }

    public abstract double yawAngleToCamera(Level var1, Camera var2, PartialTickSupplier var3);

    public abstract PitchDirection pitchDirectionToCamera(Level var1, Projector var2, PartialTickSupplier var3);

    public abstract double distanceSquared(Entity var1);

    public Waypoint.Icon icon() {
        return this.icon;
    }

    static enum Type {
        EMPTY((TriFunction<Either<UUID, String>, Waypoint.Icon, FriendlyByteBuf, TrackedWaypoint>)((TriFunction)EmptyWaypoint::new)),
        VEC3I((TriFunction<Either<UUID, String>, Waypoint.Icon, FriendlyByteBuf, TrackedWaypoint>)((TriFunction)Vec3iWaypoint::new)),
        CHUNK((TriFunction<Either<UUID, String>, Waypoint.Icon, FriendlyByteBuf, TrackedWaypoint>)((TriFunction)ChunkWaypoint::new)),
        AZIMUTH((TriFunction<Either<UUID, String>, Waypoint.Icon, FriendlyByteBuf, TrackedWaypoint>)((TriFunction)AzimuthWaypoint::new));

        final TriFunction<Either<UUID, String>, Waypoint.Icon, FriendlyByteBuf, TrackedWaypoint> constructor;

        private Type(TriFunction<Either<UUID, String>, Waypoint.Icon, FriendlyByteBuf, TrackedWaypoint> constructor) {
            this.constructor = constructor;
        }
    }

    static class Vec3iWaypoint
    extends TrackedWaypoint {
        private Vec3i vector;

        public Vec3iWaypoint(UUID uuid, Waypoint.Icon icon, Vec3i vector) {
            super(Either.left(uuid), icon, Type.VEC3I);
            this.vector = vector;
        }

        public Vec3iWaypoint(Either<UUID, String> identifier, Waypoint.Icon icon, FriendlyByteBuf buffer) {
            super(identifier, icon, Type.VEC3I);
            this.vector = new Vec3i(buffer.readVarInt(), buffer.readVarInt(), buffer.readVarInt());
        }

        @Override
        public void update(TrackedWaypoint waypoint) {
            if (waypoint instanceof Vec3iWaypoint) {
                Vec3iWaypoint vec3iWaypoint = (Vec3iWaypoint)waypoint;
                this.vector = vec3iWaypoint.vector;
            } else {
                LOGGER.warn("Unsupported Waypoint update operation: {}", waypoint.getClass());
            }
        }

        @Override
        public void writeContents(ByteBuf buffer) {
            VarInt.write(buffer, this.vector.getX());
            VarInt.write(buffer, this.vector.getY());
            VarInt.write(buffer, this.vector.getZ());
        }

        private Vec3 position(Level level, PartialTickSupplier partialTickGetter) {
            return this.identifier.left().map(level::getEntity).map(entity -> entity.blockPosition().distManhattan(this.vector) > 3 ? null : entity.getEyePosition(partialTickGetter.apply((Entity)entity))).orElseGet(() -> Vec3.atCenterOf(this.vector));
        }

        @Override
        public double yawAngleToCamera(Level level, Camera camera, PartialTickSupplier partialTickGetter) {
            Vec3 vec3 = camera.position().subtract(this.position(level, partialTickGetter)).rotateClockwise90();
            float f = (float)Mth.atan2(vec3.z(), vec3.x()) * 57.295776f;
            return Mth.degreesDifference(camera.yaw(), f);
        }

        @Override
        public PitchDirection pitchDirectionToCamera(Level level, Projector projector, PartialTickSupplier partialTickGetter) {
            double d;
            Vec3 vec3 = projector.projectPointToScreen(this.position(level, partialTickGetter));
            boolean flag = vec3.z > 1.0;
            double d2 = d = flag ? -vec3.y : vec3.y;
            if (d < -1.0) {
                return PitchDirection.DOWN;
            }
            if (d > 1.0) {
                return PitchDirection.UP;
            }
            if (flag) {
                if (vec3.y > 0.0) {
                    return PitchDirection.UP;
                }
                if (vec3.y < 0.0) {
                    return PitchDirection.DOWN;
                }
            }
            return PitchDirection.NONE;
        }

        @Override
        public double distanceSquared(Entity entity) {
            return entity.distanceToSqr(Vec3.atCenterOf(this.vector));
        }
    }

    static class ChunkWaypoint
    extends TrackedWaypoint {
        private ChunkPos chunkPos;

        public ChunkWaypoint(UUID uuid, Waypoint.Icon icon, ChunkPos chunkPos) {
            super(Either.left(uuid), icon, Type.CHUNK);
            this.chunkPos = chunkPos;
        }

        public ChunkWaypoint(Either<UUID, String> identifier, Waypoint.Icon icon, FriendlyByteBuf buffer) {
            super(identifier, icon, Type.CHUNK);
            this.chunkPos = new ChunkPos(buffer.readVarInt(), buffer.readVarInt());
        }

        @Override
        public void update(TrackedWaypoint waypoint) {
            if (waypoint instanceof ChunkWaypoint) {
                ChunkWaypoint chunkWaypoint = (ChunkWaypoint)waypoint;
                this.chunkPos = chunkWaypoint.chunkPos;
            } else {
                LOGGER.warn("Unsupported Waypoint update operation: {}", waypoint.getClass());
            }
        }

        @Override
        public void writeContents(ByteBuf buffer) {
            VarInt.write(buffer, this.chunkPos.x);
            VarInt.write(buffer, this.chunkPos.z);
        }

        private Vec3 position(double y) {
            return Vec3.atCenterOf(this.chunkPos.getMiddleBlockPosition((int)y));
        }

        @Override
        public double yawAngleToCamera(Level level, Camera camera, PartialTickSupplier partialTickGetter) {
            Vec3 vec3 = camera.position();
            Vec3 vec31 = vec3.subtract(this.position(vec3.y())).rotateClockwise90();
            float f = (float)Mth.atan2(vec31.z(), vec31.x()) * 57.295776f;
            return Mth.degreesDifference(camera.yaw(), f);
        }

        @Override
        public PitchDirection pitchDirectionToCamera(Level level, Projector projector, PartialTickSupplier partialTickGetter) {
            double d = projector.projectHorizonToScreen();
            if (d < -1.0) {
                return PitchDirection.DOWN;
            }
            return d > 1.0 ? PitchDirection.UP : PitchDirection.NONE;
        }

        @Override
        public double distanceSquared(Entity entity) {
            return entity.distanceToSqr(Vec3.atCenterOf(this.chunkPos.getMiddleBlockPosition(entity.getBlockY())));
        }
    }

    static class AzimuthWaypoint
    extends TrackedWaypoint {
        private float angle;

        public AzimuthWaypoint(UUID uuid, Waypoint.Icon icon, float angle) {
            super(Either.left(uuid), icon, Type.AZIMUTH);
            this.angle = angle;
        }

        public AzimuthWaypoint(Either<UUID, String> identifier, Waypoint.Icon icon, FriendlyByteBuf buffer) {
            super(identifier, icon, Type.AZIMUTH);
            this.angle = buffer.readFloat();
        }

        @Override
        public void update(TrackedWaypoint waypoint) {
            if (waypoint instanceof AzimuthWaypoint) {
                AzimuthWaypoint azimuthWaypoint = (AzimuthWaypoint)waypoint;
                this.angle = azimuthWaypoint.angle;
            } else {
                LOGGER.warn("Unsupported Waypoint update operation: {}", waypoint.getClass());
            }
        }

        @Override
        public void writeContents(ByteBuf buffer) {
            buffer.writeFloat(this.angle);
        }

        @Override
        public double yawAngleToCamera(Level level, Camera camera, PartialTickSupplier partialTickGetter) {
            return Mth.degreesDifference(camera.yaw(), this.angle * 57.295776f);
        }

        @Override
        public PitchDirection pitchDirectionToCamera(Level level, Projector projector, PartialTickSupplier partialTickGetter) {
            double d = projector.projectHorizonToScreen();
            if (d < -1.0) {
                return PitchDirection.DOWN;
            }
            return d > 1.0 ? PitchDirection.UP : PitchDirection.NONE;
        }

        @Override
        public double distanceSquared(Entity entity) {
            return Double.POSITIVE_INFINITY;
        }
    }

    static class EmptyWaypoint
    extends TrackedWaypoint {
        private EmptyWaypoint(Either<UUID, String> identifier, Waypoint.Icon icon, FriendlyByteBuf buffer) {
            super(identifier, icon, Type.EMPTY);
        }

        EmptyWaypoint(UUID uuid) {
            super(Either.left(uuid), Waypoint.Icon.NULL, Type.EMPTY);
        }

        @Override
        public void update(TrackedWaypoint waypoint) {
        }

        @Override
        public void writeContents(ByteBuf buffer) {
        }

        @Override
        public double yawAngleToCamera(Level level, Camera camera, PartialTickSupplier partialTickGetter) {
            return Double.NaN;
        }

        @Override
        public PitchDirection pitchDirectionToCamera(Level level, Projector projector, PartialTickSupplier partialTickGetter) {
            return PitchDirection.NONE;
        }

        @Override
        public double distanceSquared(Entity entity) {
            return Double.POSITIVE_INFINITY;
        }
    }

    public static interface Projector {
        public Vec3 projectPointToScreen(Vec3 var1);

        public double projectHorizonToScreen();
    }

    public static enum PitchDirection {
        NONE,
        UP,
        DOWN;

    }

    public static interface Camera {
        public float yaw();

        public Vec3 position();
    }
}

