/*
 * Decompiled with CFR 0.152.
 */
package net.minecraft.world.level.chunk.storage;

import com.mojang.logging.LogUtils;
import it.unimi.dsi.fastutil.longs.Long2ObjectLinkedOpenHashMap;
import java.io.IOException;
import java.nio.file.Path;
import java.util.BitSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.SequencedMap;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import java.util.function.Supplier;
import javax.annotation.Nullable;
import net.minecraft.Util;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.nbt.IntTag;
import net.minecraft.nbt.StreamTagVisitor;
import net.minecraft.nbt.Tag;
import net.minecraft.nbt.visitors.CollectFields;
import net.minecraft.nbt.visitors.FieldSelector;
import net.minecraft.util.Unit;
import net.minecraft.util.thread.PriorityConsecutiveExecutor;
import net.minecraft.util.thread.StrictQueue;
import net.minecraft.world.level.ChunkPos;
import net.minecraft.world.level.chunk.storage.ChunkScanAccess;
import net.minecraft.world.level.chunk.storage.RegionFileStorage;
import net.minecraft.world.level.chunk.storage.RegionStorageInfo;
import org.slf4j.Logger;

public class IOWorker
implements ChunkScanAccess,
AutoCloseable {
    private static final Logger LOGGER = LogUtils.getLogger();
    private final AtomicBoolean shutdownRequested = new AtomicBoolean();
    private final PriorityConsecutiveExecutor consecutiveExecutor;
    public final RegionFileStorage storage;
    private final SequencedMap<ChunkPos, PendingStore> pendingWrites = new LinkedHashMap<ChunkPos, PendingStore>();
    private final Long2ObjectLinkedOpenHashMap<CompletableFuture<BitSet>> regionCacheForBlender = new Long2ObjectLinkedOpenHashMap();
    private static final int REGION_CACHE_SIZE = 1024;

    protected IOWorker(RegionStorageInfo info, Path folder, boolean sync) {
        this.storage = new RegionFileStorage(info, folder, sync);
        this.consecutiveExecutor = new PriorityConsecutiveExecutor(Priority.values().length, (Executor)Util.ioPool(), "IOWorker-" + info.type());
    }

    public boolean isOldChunkAround(ChunkPos chunkPos, int radius) {
        ChunkPos chunkPos1 = new ChunkPos(chunkPos.x - radius, chunkPos.z - radius);
        ChunkPos chunkPos2 = new ChunkPos(chunkPos.x + radius, chunkPos.z + radius);
        for (int regionX = chunkPos1.getRegionX(); regionX <= chunkPos2.getRegionX(); ++regionX) {
            for (int regionZ = chunkPos1.getRegionZ(); regionZ <= chunkPos2.getRegionZ(); ++regionZ) {
                BitSet bitSet = this.getOrCreateOldDataForRegion(regionX, regionZ).join();
                if (bitSet.isEmpty()) continue;
                ChunkPos chunkPos3 = ChunkPos.minFromRegion(regionX, regionZ);
                int max = Math.max(chunkPos1.x - chunkPos3.x, 0);
                int max1 = Math.max(chunkPos1.z - chunkPos3.z, 0);
                int min = Math.min(chunkPos2.x - chunkPos3.x, 31);
                int min1 = Math.min(chunkPos2.z - chunkPos3.z, 31);
                for (int i = max; i <= min; ++i) {
                    for (int i1 = max1; i1 <= min1; ++i1) {
                        int i2 = i1 * 32 + i;
                        if (!bitSet.get(i2)) continue;
                        return true;
                    }
                }
            }
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private CompletableFuture<BitSet> getOrCreateOldDataForRegion(int chunkX, int chunkZ) {
        long packedChunkPos = ChunkPos.asLong(chunkX, chunkZ);
        Long2ObjectLinkedOpenHashMap<CompletableFuture<BitSet>> long2ObjectLinkedOpenHashMap = this.regionCacheForBlender;
        synchronized (long2ObjectLinkedOpenHashMap) {
            CompletableFuture<BitSet> completableFuture = (CompletableFuture<BitSet>)this.regionCacheForBlender.getAndMoveToFirst(packedChunkPos);
            if (completableFuture == null) {
                completableFuture = this.createOldDataForRegion(chunkX, chunkZ);
                this.regionCacheForBlender.putAndMoveToFirst(packedChunkPos, completableFuture);
                if (this.regionCacheForBlender.size() > 1024) {
                    this.regionCacheForBlender.removeLast();
                }
            }
            return completableFuture;
        }
    }

    private CompletableFuture<BitSet> createOldDataForRegion(int chunkX, int chunkZ) {
        return CompletableFuture.supplyAsync(() -> {
            ChunkPos chunkPos = ChunkPos.minFromRegion(chunkX, chunkZ);
            ChunkPos chunkPos1 = ChunkPos.maxFromRegion(chunkX, chunkZ);
            BitSet bitSet = new BitSet();
            ChunkPos.rangeClosed(chunkPos, chunkPos1).forEach(currentChunk -> {
                CompoundTag compoundTag;
                CollectFields collectFields = new CollectFields(new FieldSelector(IntTag.TYPE, "DataVersion"), new FieldSelector(CompoundTag.TYPE, "blending_data"));
                try {
                    this.scanChunk((ChunkPos)currentChunk, collectFields).join();
                }
                catch (Exception var7) {
                    LOGGER.warn("Failed to scan chunk {}", currentChunk, (Object)var7);
                    return;
                }
                Tag patt0$temp = collectFields.getResult();
                if (patt0$temp instanceof CompoundTag && this.isOldChunk(compoundTag = (CompoundTag)patt0$temp)) {
                    int i = currentChunk.getRegionLocalZ() * 32 + currentChunk.getRegionLocalX();
                    bitSet.set(i);
                }
            });
            return bitSet;
        }, Util.backgroundExecutor());
    }

    private boolean isOldChunk(CompoundTag chunkData) {
        return !chunkData.contains("DataVersion", 99) || chunkData.getInt("DataVersion") < 4185 || chunkData.contains("blending_data", 10);
    }

    public CompletableFuture<Void> store(ChunkPos chunkPos, @Nullable CompoundTag chunkData) {
        return this.store(chunkPos, () -> chunkData);
    }

    public CompletableFuture<Void> store(ChunkPos chunkPos, Supplier<CompoundTag> dataSupplier) {
        return this.submitTask(() -> {
            CompoundTag compoundTag = (CompoundTag)dataSupplier.get();
            PendingStore pendingStore = this.pendingWrites.computeIfAbsent(chunkPos, chunkPos1 -> new PendingStore(compoundTag));
            pendingStore.data = compoundTag;
            return pendingStore.result;
        }).thenCompose(Function.identity());
    }

    public CompletableFuture<Optional<CompoundTag>> loadAsync(ChunkPos chunkPos) {
        return this.submitThrowingTask(() -> {
            PendingStore pendingStore = (PendingStore)this.pendingWrites.get(chunkPos);
            if (pendingStore != null) {
                return Optional.ofNullable(pendingStore.copyData());
            }
            try {
                CompoundTag compoundTag = this.storage.read(chunkPos);
                return Optional.ofNullable(compoundTag);
            }
            catch (Exception var4) {
                LOGGER.warn("Failed to read chunk {}", (Object)chunkPos, (Object)var4);
                throw var4;
            }
        });
    }

    public CompletableFuture<Void> synchronize(boolean flushStorage) {
        CompletionStage completableFuture = this.submitTask(() -> CompletableFuture.allOf((CompletableFuture[])this.pendingWrites.values().stream().map(pendingStore -> pendingStore.result).toArray(CompletableFuture[]::new))).thenCompose(Function.identity());
        return flushStorage ? ((CompletableFuture)completableFuture).thenCompose(_void -> this.submitThrowingTask(() -> {
            try {
                this.storage.flush();
                return null;
            }
            catch (Exception var2x) {
                LOGGER.warn("Failed to synchronize chunks", (Throwable)var2x);
                throw var2x;
            }
        })) : ((CompletableFuture)completableFuture).thenCompose(_void -> this.submitTask(() -> null));
    }

    @Override
    public CompletableFuture<Void> scanChunk(ChunkPos chunkPos, StreamTagVisitor visitor) {
        return this.submitThrowingTask(() -> {
            try {
                PendingStore pendingStore = (PendingStore)this.pendingWrites.get(chunkPos);
                if (pendingStore != null) {
                    if (pendingStore.data != null) {
                        pendingStore.data.acceptAsRoot(visitor);
                    }
                } else {
                    this.storage.scanChunk(chunkPos, visitor);
                }
                return null;
            }
            catch (Exception var4) {
                LOGGER.warn("Failed to bulk scan chunk {}", (Object)chunkPos, (Object)var4);
                throw var4;
            }
        });
    }

    private <T> CompletableFuture<T> submitThrowingTask(ThrowingSupplier<T> task) {
        return this.consecutiveExecutor.scheduleWithResult(Priority.FOREGROUND.ordinal(), completableFuture -> {
            if (!this.shutdownRequested.get()) {
                try {
                    completableFuture.complete(task.get());
                }
                catch (Exception var4) {
                    completableFuture.completeExceptionally(var4);
                }
            }
            this.tellStorePending();
        });
    }

    private <T> CompletableFuture<T> submitTask(Supplier<T> task) {
        return this.consecutiveExecutor.scheduleWithResult(Priority.FOREGROUND.ordinal(), completableFuture -> {
            if (!this.shutdownRequested.get()) {
                completableFuture.complete(task.get());
            }
            this.tellStorePending();
        });
    }

    private void storePendingChunk() {
        Map.Entry<ChunkPos, PendingStore> entry = this.pendingWrites.pollFirstEntry();
        if (entry != null) {
            this.runStore(entry.getKey(), entry.getValue());
            this.tellStorePending();
        }
    }

    private void tellStorePending() {
        this.consecutiveExecutor.schedule(new StrictQueue.RunnableWithPriority(Priority.BACKGROUND.ordinal(), this::storePendingChunk));
    }

    private void runStore(ChunkPos chunkPos, PendingStore pendingStore) {
        try {
            this.storage.write(chunkPos, pendingStore.data);
            pendingStore.result.complete(null);
        }
        catch (Exception var4) {
            LOGGER.error("Failed to store chunk {}", (Object)chunkPos, (Object)var4);
            pendingStore.result.completeExceptionally(var4);
        }
    }

    @Override
    public void close() throws IOException {
        if (this.shutdownRequested.compareAndSet(false, true)) {
            this.waitForShutdown();
            this.consecutiveExecutor.close();
            try {
                this.storage.close();
            }
            catch (Exception var2) {
                LOGGER.error("Failed to close storage", (Throwable)var2);
            }
        }
    }

    private void waitForShutdown() {
        this.consecutiveExecutor.scheduleWithResult(Priority.SHUTDOWN.ordinal(), completableFuture -> completableFuture.complete(Unit.INSTANCE)).join();
    }

    public RegionStorageInfo storageInfo() {
        return this.storage.info();
    }

    static enum Priority {
        FOREGROUND,
        BACKGROUND,
        SHUTDOWN;

    }

    @FunctionalInterface
    static interface ThrowingSupplier<T> {
        @Nullable
        public T get() throws Exception;
    }

    static class PendingStore {
        @Nullable
        CompoundTag data;
        final CompletableFuture<Void> result = new CompletableFuture();

        public PendingStore(@Nullable CompoundTag data) {
            this.data = data;
        }

        @Nullable
        CompoundTag copyData() {
            CompoundTag compoundTag = this.data;
            return compoundTag == null ? null : compoundTag.copy();
        }
    }
}

