/*
 * Decompiled with CFR 0.152.
 */
package net.minecraft.world.level.storage;

import ca.spottedleaf.dataconverter.minecraft.MCDataConverter;
import ca.spottedleaf.dataconverter.minecraft.datatypes.MCTypeRegistry;
import com.google.common.collect.Maps;
import com.mojang.datafixers.DataFixer;
import com.mojang.logging.LogUtils;
import com.mojang.serialization.Dynamic;
import com.mojang.serialization.Lifecycle;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import javax.annotation.Nullable;
import net.minecraft.CrashReport;
import net.minecraft.CrashReportCategory;
import net.minecraft.FileUtil;
import net.minecraft.ReportedException;
import net.minecraft.SharedConstants;
import net.minecraft.Util;
import net.minecraft.core.HolderLookup;
import net.minecraft.core.Registry;
import net.minecraft.core.RegistryAccess;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.nbt.NbtAccounter;
import net.minecraft.nbt.NbtFormatException;
import net.minecraft.nbt.NbtIo;
import net.minecraft.nbt.NbtOps;
import net.minecraft.nbt.NbtUtils;
import net.minecraft.nbt.StreamTagVisitor;
import net.minecraft.nbt.Tag;
import net.minecraft.nbt.visitors.FieldSelector;
import net.minecraft.nbt.visitors.SkipFields;
import net.minecraft.network.chat.Component;
import net.minecraft.resources.RegistryOps;
import net.minecraft.resources.ResourceKey;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.WorldLoader;
import net.minecraft.server.packs.repository.PackRepository;
import net.minecraft.util.DirectoryLock;
import net.minecraft.util.MemoryReserve;
import net.minecraft.util.datafix.DataFixTypes;
import net.minecraft.util.datafix.DataFixers;
import net.minecraft.world.flag.FeatureFlagSet;
import net.minecraft.world.flag.FeatureFlags;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.LevelSettings;
import net.minecraft.world.level.WorldDataConfiguration;
import net.minecraft.world.level.dimension.LevelStem;
import net.minecraft.world.level.levelgen.WorldDimensions;
import net.minecraft.world.level.levelgen.WorldGenSettings;
import net.minecraft.world.level.storage.FileNameDateFormatter;
import net.minecraft.world.level.storage.LevelDataAndDimensions;
import net.minecraft.world.level.storage.LevelResource;
import net.minecraft.world.level.storage.LevelStorageException;
import net.minecraft.world.level.storage.LevelSummary;
import net.minecraft.world.level.storage.LevelVersion;
import net.minecraft.world.level.storage.PlayerDataStorage;
import net.minecraft.world.level.storage.PrimaryLevelData;
import net.minecraft.world.level.storage.WorldData;
import net.minecraft.world.level.validation.ContentValidationException;
import net.minecraft.world.level.validation.DirectoryValidator;
import net.minecraft.world.level.validation.ForbiddenSymlinkInfo;
import net.minecraft.world.level.validation.PathAllowList;
import org.slf4j.Logger;

public class LevelStorageSource {
    static final Logger LOGGER = LogUtils.getLogger();
    static final DateTimeFormatter FORMATTER = FileNameDateFormatter.create();
    public static final String TAG_DATA = "Data";
    private static final PathMatcher NO_SYMLINKS_ALLOWED = path -> false;
    public static final String ALLOWED_SYMLINKS_CONFIG_NAME = "allowed_symlinks.txt";
    private static final int UNCOMPRESSED_NBT_QUOTA = 0x6400000;
    private static final int DISK_SPACE_WARNING_THRESHOLD = 0x4000000;
    public final Path baseDir;
    private final Path backupDir;
    final DataFixer fixerUpper;
    private final DirectoryValidator worldDirValidator;

    public LevelStorageSource(Path baseDir, Path backupDir, DirectoryValidator worldDirValidator, DataFixer fixerUpper) {
        this.fixerUpper = fixerUpper;
        try {
            FileUtil.createDirectoriesSafe(baseDir);
        }
        catch (IOException var6) {
            throw new UncheckedIOException(var6);
        }
        this.baseDir = baseDir;
        this.backupDir = backupDir;
        this.worldDirValidator = worldDirValidator;
    }

    public static DirectoryValidator parseValidator(Path validator) {
        if (Files.exists(validator, new LinkOption[0])) {
            try {
                DirectoryValidator var2;
                try (BufferedReader bufferedReader = Files.newBufferedReader(validator);){
                    var2 = new DirectoryValidator(PathAllowList.readPlain(bufferedReader));
                }
                return var2;
            }
            catch (Exception var6) {
                LOGGER.error("Failed to parse {}, disallowing all symbolic links", (Object)ALLOWED_SYMLINKS_CONFIG_NAME, (Object)var6);
            }
        }
        return new DirectoryValidator(NO_SYMLINKS_ALLOWED);
    }

    public static LevelStorageSource createDefault(Path savesDir) {
        DirectoryValidator directoryValidator = LevelStorageSource.parseValidator(savesDir.resolve(ALLOWED_SYMLINKS_CONFIG_NAME));
        return new LevelStorageSource(savesDir, savesDir.resolve("../backups"), directoryValidator, DataFixers.getDataFixer());
    }

    public static WorldDataConfiguration readDataConfig(Dynamic<?> dynamic) {
        return WorldDataConfiguration.CODEC.parse(dynamic).resultOrPartial(arg_0 -> ((Logger)LOGGER).error(arg_0)).orElse(WorldDataConfiguration.DEFAULT);
    }

    public static WorldLoader.PackConfig getPackConfig(Dynamic<?> dynamic, PackRepository packRepository, boolean safeMode) {
        return new WorldLoader.PackConfig(packRepository, LevelStorageSource.readDataConfig(dynamic), safeMode, false);
    }

    public static LevelDataAndDimensions getLevelDataAndDimensions(Dynamic<?> levelData, WorldDataConfiguration dataConfiguration, Registry<LevelStem> levelStemRegistry, HolderLookup.Provider registries) {
        Dynamic<?> dynamic = RegistryOps.injectRegistryContext(levelData, registries);
        Dynamic dynamic1 = dynamic.get("WorldGenSettings").orElseEmptyMap();
        WorldGenSettings worldGenSettings = (WorldGenSettings)WorldGenSettings.CODEC.parse(dynamic1).getOrThrow();
        LevelSettings levelSettings = LevelSettings.parse(dynamic, dataConfiguration);
        WorldDimensions.Complete complete = worldGenSettings.dimensions().bake(levelStemRegistry);
        Lifecycle lifecycle = complete.lifecycle().add(registries.allRegistriesLifecycle());
        PrimaryLevelData primaryLevelData = PrimaryLevelData.parse(dynamic, levelSettings, complete.specialWorldProperty(), worldGenSettings.options(), lifecycle);
        primaryLevelData.pdc = (Tag)dynamic.getElement("BukkitValues", null);
        return new LevelDataAndDimensions(primaryLevelData, complete);
    }

    public String getName() {
        return "Anvil";
    }

    public LevelCandidates findLevelCandidates() throws LevelStorageException {
        if (!Files.isDirectory(this.baseDir, new LinkOption[0])) {
            throw new LevelStorageException(Component.translatable("selectWorld.load_folder_access"));
        }
        try {
            LevelCandidates var3;
            try (Stream<Path> stream = Files.list(this.baseDir);){
                List<LevelDirectory> list = stream.filter(path -> Files.isDirectory(path, new LinkOption[0])).map(LevelDirectory::new).filter(levelDirectory -> Files.isRegularFile(levelDirectory.dataFile(), new LinkOption[0]) || Files.isRegularFile(levelDirectory.oldDataFile(), new LinkOption[0])).toList();
                var3 = new LevelCandidates(list);
            }
            return var3;
        }
        catch (IOException var6) {
            throw new LevelStorageException(Component.translatable("selectWorld.load_folder_access"));
        }
    }

    public CompletableFuture<List<LevelSummary>> loadLevelSummaries(LevelCandidates candidates) {
        ArrayList<CompletableFuture<LevelSummary>> list = new ArrayList<CompletableFuture<LevelSummary>>(candidates.levels.size());
        for (LevelDirectory levelDirectory : candidates.levels) {
            list.add(CompletableFuture.supplyAsync(() -> {
                boolean isLocked;
                try {
                    isLocked = DirectoryLock.isLocked(levelDirectory.path());
                }
                catch (Exception var13) {
                    LOGGER.warn("Failed to read {} lock", (Object)levelDirectory.path(), (Object)var13);
                    return null;
                }
                try {
                    return this.readLevelSummary(levelDirectory, isLocked);
                }
                catch (OutOfMemoryError var12) {
                    MemoryReserve.release();
                    String string = "Ran out of memory trying to read summary of world folder \"" + levelDirectory.directoryName() + "\"";
                    LOGGER.error(LogUtils.FATAL_MARKER, string);
                    OutOfMemoryError outOfMemoryError1 = new OutOfMemoryError("Ran out of memory reading level data");
                    outOfMemoryError1.initCause(var12);
                    CrashReport crashReport = CrashReport.forThrowable(outOfMemoryError1, string);
                    CrashReportCategory crashReportCategory = crashReport.addCategory("World details");
                    crashReportCategory.setDetail("Folder Name", levelDirectory.directoryName());
                    try {
                        long size = Files.size(levelDirectory.dataFile());
                        crashReportCategory.setDetail("level.dat size", size);
                    }
                    catch (IOException var11) {
                        crashReportCategory.setDetailError("level.dat size", var11);
                    }
                    throw new ReportedException(crashReport);
                }
            }, Util.backgroundExecutor().forName("loadLevelSummaries")));
        }
        return Util.sequenceFailFastAndCancel(list).thenApply(list1 -> list1.stream().filter(Objects::nonNull).sorted().toList());
    }

    private int getStorageVersion() {
        return 19133;
    }

    static CompoundTag readLevelDataTagRaw(Path levelPath) throws IOException {
        return NbtIo.readCompressed(levelPath, NbtAccounter.create(0x6400000L));
    }

    static Dynamic<?> readLevelDataTagFixed(Path levelPath, DataFixer dataFixer) throws IOException {
        CompoundTag levelDataTagRaw = LevelStorageSource.readLevelDataTagRaw(levelPath);
        CompoundTag compound = levelDataTagRaw.getCompound(TAG_DATA);
        int dataVersion = NbtUtils.getDataVersion(compound, -1);
        Dynamic<CompoundTag> dynamic = DataFixTypes.LEVEL.updateToCurrentVersion(dataFixer, new Dynamic<CompoundTag>(NbtOps.INSTANCE, compound), dataVersion);
        dynamic = dynamic.update("Player", dynamic1 -> new Dynamic<CompoundTag>(dynamic1.getOps(), MCDataConverter.convertTag(MCTypeRegistry.PLAYER, (CompoundTag)dynamic1.getValue(), dataVersion, SharedConstants.getCurrentVersion().getDataVersion().getVersion())));
        return dynamic.update("WorldGenSettings", dynamic1 -> DataFixTypes.WORLD_GEN_SETTINGS.updateToCurrentVersion(dataFixer, dynamic1, dataVersion));
    }

    private LevelSummary readLevelSummary(LevelDirectory levelDirectory, boolean locked) {
        Path path = levelDirectory.dataFile();
        if (Files.exists(path, new LinkOption[0])) {
            try {
                List<ForbiddenSymlinkInfo> list;
                if (Files.isSymbolicLink(path) && !(list = this.worldDirValidator.validateSymlink(path)).isEmpty()) {
                    LOGGER.warn("{}", (Object)ContentValidationException.getMessage(path, list));
                    return new LevelSummary.SymlinkLevelSummary(levelDirectory.directoryName(), levelDirectory.iconFile());
                }
                Tag tag = LevelStorageSource.readLightweightData(path);
                if (tag instanceof CompoundTag) {
                    CompoundTag compoundTag = (CompoundTag)tag;
                    CompoundTag compound = compoundTag.getCompound(TAG_DATA);
                    int dataVersion = NbtUtils.getDataVersion(compound, -1);
                    Dynamic<CompoundTag> dynamic = DataFixTypes.LEVEL.updateToCurrentVersion(this.fixerUpper, new Dynamic<CompoundTag>(NbtOps.INSTANCE, compound), dataVersion);
                    return this.makeLevelSummary(dynamic, levelDirectory, locked);
                }
                LOGGER.warn("Invalid root tag in {}", (Object)path);
            }
            catch (Exception var9) {
                LOGGER.error("Exception reading {}", (Object)path, (Object)var9);
            }
        }
        return new LevelSummary.CorruptedLevelSummary(levelDirectory.directoryName(), levelDirectory.iconFile(), LevelStorageSource.getFileModificationTime(levelDirectory));
    }

    private static long getFileModificationTime(LevelDirectory levelDirectory) {
        Instant fileModificationTime = LevelStorageSource.getFileModificationTime(levelDirectory.dataFile());
        if (fileModificationTime == null) {
            fileModificationTime = LevelStorageSource.getFileModificationTime(levelDirectory.oldDataFile());
        }
        return fileModificationTime == null ? -1L : fileModificationTime.toEpochMilli();
    }

    @Nullable
    static Instant getFileModificationTime(Path dataFilePath) {
        try {
            return Files.getLastModifiedTime(dataFilePath, new LinkOption[0]).toInstant();
        }
        catch (IOException var2) {
            return null;
        }
    }

    LevelSummary makeLevelSummary(Dynamic<?> dynamic, LevelDirectory levelDirectory, boolean locked) {
        LevelVersion levelVersion = LevelVersion.parse(dynamic);
        int levelDataVersion = levelVersion.levelDataVersion();
        if (levelDataVersion != 19132 && levelDataVersion != 19133) {
            throw new NbtFormatException("Unknown data version: " + Integer.toHexString(levelDataVersion));
        }
        boolean flag = levelDataVersion != this.getStorageVersion();
        Path path = levelDirectory.iconFile();
        WorldDataConfiguration dataConfig = LevelStorageSource.readDataConfig(dynamic);
        LevelSettings levelSettings = LevelSettings.parse(dynamic, dataConfig);
        FeatureFlagSet featureFlagSet = LevelStorageSource.parseFeatureFlagsFromSummary(dynamic);
        boolean isExperimental = FeatureFlags.isExperimental(featureFlagSet);
        return new LevelSummary(levelSettings, levelVersion, levelDirectory.directoryName(), flag, locked, isExperimental, path);
    }

    private static FeatureFlagSet parseFeatureFlagsFromSummary(Dynamic<?> dataDynamic) {
        Set<ResourceLocation> set = dataDynamic.get("enabled_features").asStream().flatMap(dynamic -> dynamic.asString().result().map(ResourceLocation::tryParse).stream()).collect(Collectors.toSet());
        return FeatureFlags.REGISTRY.fromNames(set, resourceLocation -> {});
    }

    @Nullable
    private static Tag readLightweightData(Path file) throws IOException {
        SkipFields skipFields = new SkipFields(new FieldSelector(TAG_DATA, CompoundTag.TYPE, "Player"), new FieldSelector(TAG_DATA, CompoundTag.TYPE, "WorldGenSettings"));
        NbtIo.parseCompressed(file, (StreamTagVisitor)skipFields, NbtAccounter.create(0x6400000L));
        return skipFields.getResult();
    }

    public boolean isNewLevelIdAcceptable(String saveName) {
        try {
            Path levelPath = this.getLevelPath(saveName);
            Files.createDirectory(levelPath, new FileAttribute[0]);
            Files.deleteIfExists(levelPath);
            return true;
        }
        catch (IOException var3) {
            return false;
        }
    }

    public boolean levelExists(String saveName) {
        try {
            return Files.isDirectory(this.getLevelPath(saveName), new LinkOption[0]);
        }
        catch (InvalidPathException var3) {
            return false;
        }
    }

    public Path getLevelPath(String saveName) {
        return this.baseDir.resolve(saveName);
    }

    public Path getBaseDir() {
        return this.baseDir;
    }

    public Path getBackupPath() {
        return this.backupDir;
    }

    public LevelStorageAccess validateAndCreateAccess(String saveName, ResourceKey<LevelStem> dimensionType) throws IOException, ContentValidationException {
        List<ForbiddenSymlinkInfo> list;
        Path levelPath = this.getLevelPath(saveName);
        List<ForbiddenSymlinkInfo> list2 = list = Boolean.getBoolean("paper.disableWorldSymlinkValidation") ? List.of() : this.worldDirValidator.validateDirectory(levelPath, true);
        if (!list.isEmpty()) {
            throw new ContentValidationException(levelPath, list);
        }
        return new LevelStorageAccess(saveName, levelPath, dimensionType);
    }

    public LevelStorageAccess createAccess(String saveName, ResourceKey<LevelStem> dimensionType) throws IOException {
        Path levelPath = this.getLevelPath(saveName);
        return new LevelStorageAccess(saveName, levelPath, dimensionType);
    }

    public DirectoryValidator getWorldDirValidator() {
        return this.worldDirValidator;
    }

    public static Path getStorageFolder(Path path, ResourceKey<LevelStem> dimensionType) {
        if (dimensionType == LevelStem.OVERWORLD) {
            return path;
        }
        if (dimensionType == LevelStem.NETHER) {
            return path.resolve("DIM-1");
        }
        if (dimensionType == LevelStem.END) {
            return path.resolve("DIM1");
        }
        return path.resolve("dimensions").resolve(dimensionType.location().getNamespace()).resolve(dimensionType.location().getPath());
    }

    public record LevelCandidates(List<LevelDirectory> levels) implements Iterable<LevelDirectory>
    {
        public boolean isEmpty() {
            return this.levels.isEmpty();
        }

        @Override
        public Iterator<LevelDirectory> iterator() {
            return this.levels.iterator();
        }
    }

    public record LevelDirectory(Path path) {
        public String directoryName() {
            return this.path.getFileName().toString();
        }

        public Path dataFile() {
            return this.resourcePath(LevelResource.LEVEL_DATA_FILE);
        }

        public Path oldDataFile() {
            return this.resourcePath(LevelResource.OLD_LEVEL_DATA_FILE);
        }

        public Path corruptedDataFile(LocalDateTime dateTime) {
            return this.path.resolve(LevelResource.LEVEL_DATA_FILE.getId() + "_corrupted_" + dateTime.format(FORMATTER));
        }

        public Path rawDataFile(LocalDateTime dateTime) {
            return this.path.resolve(LevelResource.LEVEL_DATA_FILE.getId() + "_raw_" + dateTime.format(FORMATTER));
        }

        public Path iconFile() {
            return this.resourcePath(LevelResource.ICON_FILE);
        }

        public Path lockFile() {
            return this.resourcePath(LevelResource.LOCK_FILE);
        }

        public Path resourcePath(LevelResource resource) {
            return this.path.resolve(resource.getId());
        }
    }

    public class LevelStorageAccess
    implements AutoCloseable {
        final DirectoryLock lock;
        public final LevelDirectory levelDirectory;
        private final String levelId;
        private final Map<LevelResource, Path> resources = Maps.newHashMap();
        public final ResourceKey<LevelStem> dimensionType;

        LevelStorageAccess(String levelId, Path levelDir, ResourceKey<LevelStem> dimensionType) throws IOException {
            this.dimensionType = dimensionType;
            this.levelId = levelId;
            this.levelDirectory = new LevelDirectory(levelDir);
            this.lock = DirectoryLock.create(levelDir);
        }

        public long estimateDiskSpace() {
            try {
                return Files.getFileStore(this.levelDirectory.path).getUsableSpace();
            }
            catch (Exception var2) {
                return Long.MAX_VALUE;
            }
        }

        public boolean checkForLowDiskSpace() {
            return this.estimateDiskSpace() < 0x4000000L;
        }

        public void safeClose() {
            try {
                this.close();
            }
            catch (IOException var2) {
                LOGGER.warn("Failed to unlock access to level {}", (Object)this.getLevelId(), (Object)var2);
            }
        }

        public LevelStorageSource parent() {
            return LevelStorageSource.this;
        }

        public LevelDirectory getLevelDirectory() {
            return this.levelDirectory;
        }

        public String getLevelId() {
            return this.levelId;
        }

        public Path getLevelPath(LevelResource folderName) {
            return this.resources.computeIfAbsent(folderName, this.levelDirectory::resourcePath);
        }

        public Path getDimensionPath(ResourceKey<Level> dimensionPath) {
            return LevelStorageSource.getStorageFolder(this.levelDirectory.path(), this.dimensionType);
        }

        private void checkLock() {
            if (!this.lock.isValid()) {
                throw new IllegalStateException("Lock is no longer valid");
            }
        }

        public PlayerDataStorage createPlayerStorage() {
            this.checkLock();
            return new PlayerDataStorage(this, LevelStorageSource.this.fixerUpper);
        }

        public LevelSummary getSummary(Dynamic<?> dynamic) {
            this.checkLock();
            return LevelStorageSource.this.makeLevelSummary(dynamic, this.levelDirectory, false);
        }

        public Dynamic<?> getDataTag() throws IOException {
            return this.getDataTag(false);
        }

        public Dynamic<?> getDataTagFallback() throws IOException {
            return this.getDataTag(true);
        }

        private Dynamic<?> getDataTag(boolean useFallback) throws IOException {
            this.checkLock();
            return LevelStorageSource.readLevelDataTagFixed(useFallback ? this.levelDirectory.oldDataFile() : this.levelDirectory.dataFile(), LevelStorageSource.this.fixerUpper);
        }

        public void saveDataTag(RegistryAccess registries, WorldData serverConfiguration) {
            this.saveDataTag(registries, serverConfiguration, null);
        }

        public void saveDataTag(RegistryAccess registries, WorldData serverConfiguration, @Nullable CompoundTag hostPlayerNBT) {
            CompoundTag compoundTag = serverConfiguration.createTag(registries, hostPlayerNBT);
            CompoundTag compoundTag1 = new CompoundTag();
            compoundTag1.put(LevelStorageSource.TAG_DATA, compoundTag);
            this.saveLevelData(compoundTag1);
        }

        private void saveLevelData(CompoundTag tag) {
            Path path = this.levelDirectory.path();
            try {
                Path path1 = Files.createTempFile(path, "level", ".dat", new FileAttribute[0]);
                NbtIo.writeCompressed(tag, path1);
                Path path2 = this.levelDirectory.oldDataFile();
                Path path3 = this.levelDirectory.dataFile();
                Util.safeReplaceFile(path3, path1, path2);
            }
            catch (Exception var6) {
                LOGGER.error("Failed to save level {}", (Object)path, (Object)var6);
            }
        }

        public Optional<Path> getIconFile() {
            return !this.lock.isValid() ? Optional.empty() : Optional.of(this.levelDirectory.iconFile());
        }

        public void deleteLevel() throws IOException {
            this.checkLock();
            final Path path = this.levelDirectory.lockFile();
            LOGGER.info("Deleting level {}", (Object)this.levelId);
            for (int i = 1; i <= 5; ++i) {
                LOGGER.info("Attempt {}...", (Object)i);
                try {
                    Files.walkFileTree(this.levelDirectory.path(), (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                        @Override
                        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                            if (!file.equals(path)) {
                                LOGGER.debug("Deleting {}", (Object)file);
                                Files.delete(file);
                            }
                            return FileVisitResult.CONTINUE;
                        }

                        @Override
                        public FileVisitResult postVisitDirectory(Path dir, @Nullable IOException exception) throws IOException {
                            if (exception != null) {
                                throw exception;
                            }
                            if (dir.equals(LevelStorageAccess.this.levelDirectory.path())) {
                                LevelStorageAccess.this.lock.close();
                                Files.deleteIfExists(path);
                            }
                            Files.delete(dir);
                            return FileVisitResult.CONTINUE;
                        }
                    });
                    break;
                }
                catch (IOException var6) {
                    if (i >= 5) {
                        throw var6;
                    }
                    LOGGER.warn("Failed to delete {}", (Object)this.levelDirectory.path(), (Object)var6);
                    try {
                        Thread.sleep(500L);
                    }
                    catch (InterruptedException interruptedException) {
                        // empty catch block
                    }
                    continue;
                }
            }
        }

        public void renameLevel(String saveName) throws IOException {
            this.modifyLevelDataWithoutDatafix(compoundTag -> compoundTag.putString("LevelName", saveName.trim()));
        }

        public void renameAndDropPlayer(String saveName) throws IOException {
            this.modifyLevelDataWithoutDatafix(compoundTag -> {
                compoundTag.putString("LevelName", saveName.trim());
                compoundTag.remove("Player");
            });
        }

        private void modifyLevelDataWithoutDatafix(Consumer<CompoundTag> modifier) throws IOException {
            this.checkLock();
            CompoundTag levelDataTagRaw = LevelStorageSource.readLevelDataTagRaw(this.levelDirectory.dataFile());
            modifier.accept(levelDataTagRaw.getCompound(LevelStorageSource.TAG_DATA));
            this.saveLevelData(levelDataTagRaw);
        }

        public long makeWorldBackup() throws IOException {
            this.checkLock();
            String string = LocalDateTime.now().format(FORMATTER) + "_" + this.levelId;
            Path backupPath = LevelStorageSource.this.getBackupPath();
            try {
                FileUtil.createDirectoriesSafe(backupPath);
            }
            catch (IOException var9) {
                throw new RuntimeException(var9);
            }
            Path path = backupPath.resolve(FileUtil.findAvailableName(backupPath, string, ".zip"));
            try (final ZipOutputStream zipOutputStream = new ZipOutputStream(new BufferedOutputStream(Files.newOutputStream(path, new OpenOption[0])));){
                final Path path1 = Paths.get(this.levelId, new String[0]);
                Files.walkFileTree(this.levelDirectory.path(), (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                    @Override
                    public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                        if (file.endsWith("session.lock")) {
                            return FileVisitResult.CONTINUE;
                        }
                        String string1 = path1.resolve(LevelStorageAccess.this.levelDirectory.path().relativize(file)).toString().replace('\\', '/');
                        ZipEntry zipEntry = new ZipEntry(string1);
                        zipOutputStream.putNextEntry(zipEntry);
                        com.google.common.io.Files.asByteSource((File)file.toFile()).copyTo((OutputStream)zipOutputStream);
                        zipOutputStream.closeEntry();
                        return FileVisitResult.CONTINUE;
                    }
                });
            }
            return Files.size(path);
        }

        public boolean hasWorldData() {
            return Files.exists(this.levelDirectory.dataFile(), new LinkOption[0]) || Files.exists(this.levelDirectory.oldDataFile(), new LinkOption[0]);
        }

        @Override
        public void close() throws IOException {
            this.lock.close();
        }

        public boolean restoreLevelDataFromOld() {
            return Util.safeReplaceOrMoveFile(this.levelDirectory.dataFile(), this.levelDirectory.oldDataFile(), this.levelDirectory.corruptedDataFile(LocalDateTime.now()), true);
        }

        @Nullable
        public Instant getFileModificationTime(boolean useFallback) {
            return LevelStorageSource.getFileModificationTime(useFallback ? this.levelDirectory.oldDataFile() : this.levelDirectory.dataFile());
        }
    }
}

